#' ggbiplot script
#' @Author: Vince Q. Vu: vqv@stat.osu.edu
#' This function generates PCA biplots
ggbiplot <- function (pcobj, choices = 1:2, scale = 1, pc.biplot = TRUE, 
                      obs.scale = 1 - scale, var.scale = scale, groups = NULL, 
                      ellipse = FALSE, ellipse.prob = 0.68, labels = NULL, labels.size = 3, 
                      alpha = 1, var.axes = TRUE, circle = FALSE, circle.prob = 0.69, 
                      varname.size = 3, varname.adjust = 1.5, varname.abbrev = FALSE, 
                      point.size = 3,widen.axes.by = 1.2,
                      ...) 
{
  library(ggplot2)
  library(plyr)
  library(scales)
  library(grid)
  stopifnot(length(choices) == 2)
  if (inherits(pcobj, "prcomp")) {
    nobs.factor <- sqrt(nrow(pcobj$x) - 1)
    d <- pcobj$sdev
    u <- sweep(pcobj$x, 2, 1/(d * nobs.factor), FUN = "*")
    v <- pcobj$rotation
  }
  else if (inherits(pcobj, "princomp")) {
    nobs.factor <- sqrt(pcobj$n.obs)
    d <- pcobj$sdev
    u <- sweep(pcobj$scores, 2, 1/(d * nobs.factor), FUN = "*")
    v <- pcobj$loadings
  }
  else if (inherits(pcobj, "PCA")) {
    nobs.factor <- sqrt(nrow(pcobj$call$X))
    d <- unlist(sqrt(pcobj$eig)[1])
    u <- sweep(pcobj$ind$coord, 2, 1/(d * nobs.factor), FUN = "*")
    v <- sweep(pcobj$var$coord, 2, sqrt(pcobj$eig[1:ncol(pcobj$var$coord), 
                                                  1]), FUN = "/")
  }
  else if (inherits(pcobj, "lda")) {
    nobs.factor <- sqrt(pcobj$N)
    d <- pcobj$svd
    u <- predict(pcobj)$x/nobs.factor
    v <- pcobj$scaling
    d.total <- sum(d^2)
  }
  else {
    stop("Expected a object of class prcomp, princomp, PCA, or lda")
  }
  choices <- pmin(choices, ncol(u))
  df.u <- as.data.frame(sweep(u[, choices], 2, d[choices]^obs.scale,  FUN = "*"))
  v <- sweep(v, 2, d^var.scale, FUN = "*")
  df.v <- as.data.frame(v[, choices])
  names(df.u) <- c("xvar", "yvar")
  names(df.v) <- names(df.u)
  if (pc.biplot) {
    df.u <- df.u * nobs.factor
  }
  r <- sqrt(qchisq(circle.prob, df = 2)) * prod(colMeans(df.u^2))^(1/4)
  v.scale <- rowSums(v^2)
  df.v <- r * df.v/sqrt(max(v.scale))
  if (obs.scale == 0) {
    u.axis.labs <- paste("standardized PC", choices, sep = "")
  }
  else {
    u.axis.labs <- paste("PC", choices, sep = "")
  }
  u.axis.labs <- paste(u.axis.labs, sprintf("(%0.1f%% explained var.)", 
                                            100 * pcobj$sdev[choices]^2/sum(pcobj$sdev^2)))
  if (!is.null(labels)) {
    df.u$labels <- labels
  }
  if (!is.null(groups)) {
    df.u$groups <- groups
  }
  if (varname.abbrev) {
    df.v$varname <- abbreviate(rownames(v))
  }
  else {
    df.v$varname <- rownames(v)
  }
  df.v$angle <- with(df.v, (180/pi) * atan(yvar/xvar))
  df.v$hjust = with(df.v, (1 - varname.adjust * sign(xvar))/2)
  
  
  g <- ggplot(data = df.u, aes(x = xvar, y = yvar)) + 
    xlab(u.axis.labs[1]) + 
    ylab(u.axis.labs[2]) + 
    coord_equal()
  
  #Add 20% to the X and Y scale
  # g.built <- ggplot_build(g)
  # Lx <- diff(g.built$panel$ranges[[1]]$x.range)*widen.axes.by
  # Ly <- diff(g.built$panel$ranges[[1]]$y.range)*widen.axes.by
  # centx <- min(g.built$panel$ranges[[1]]$x.range) + diff(g.built$panel$ranges[[1]]$x.range)/2
  # centy <- min(g.built$panel$ranges[[1]]$y.range) + diff(g.built$panel$ranges[[1]]$y.range)/2
  # 
  # g <- g + 
  #   scale_x_continuous(limits = centx + c(-Lx/2, Lx/2)) +
  #   scale_y_continuous(limits = centy + c(-Ly/2, Ly/2))
  
  if (var.axes) {
    if (circle) {
      theta <- c(seq(-pi, pi, length = 50), seq(pi, -pi, 
                                                length = 50))
      circle <- data.frame(xvar = r * cos(theta), yvar = r * 
                             sin(theta))
      g <- g + geom_path(data = circle, color = muted("white"), 
                         size = 1/2, alpha = 1/3)
    }
    g <- g + geom_segment(data = df.v, aes(x = 0, y = 0, xend = xvar, yend = yvar),
                          arrow = arrow(length = unit(1/2, "picas")),
                          color = muted("red"))
  }
  if (!is.null(df.u$labels)) {
    if (!is.null(df.u$groups)) {
      g <- g + geom_text(aes(label = labels, color = groups), 
                         size = labels.size)
    }
    else {
      g <- g + geom_text(aes(label = labels), size = labels.size)
    }
  }
  else {
    if (!is.null(df.u$groups)) {
      g <- g + geom_point(aes(color = groups), size = point.size,alpha = alpha)
    }
    else {
      g <- g + geom_point(alpha = alpha, size = point.size)
    }
  }
  if (!is.null(df.u$groups) && ellipse) {
    theta <- c(seq(-pi, pi, length = 50), seq(pi, -pi, length = 50))
    circle <- cbind(cos(theta), sin(theta))
    ell <- ddply(df.u, "groups", function(x) {
      if (nrow(x) <= 2) {
        return(NULL)
      }
      sigma <- var(cbind(x$xvar, x$yvar))
      mu <- c(mean(x$xvar), mean(x$yvar))
      ed <- sqrt(qchisq(ellipse.prob, df = 2))
      data.frame(sweep(circle %*% chol(sigma) * ed, 2, 
                       mu, FUN = "+"), groups = x$groups[1])
    })
    names(ell)[1:2] <- c("xvar", "yvar")
    g <- g + geom_path(data = ell, aes(color = groups, group = groups))
  }
  if (var.axes) {
    
    # Check if the var.axes labels are likely to be beyond the limits
    ggplot_build(g)->g.built
    quantiles.of.limits <- abs(c(df.v$xvar/max(g.built$panel$ranges[[1]]$x.range),
                               df.v$xvar/min(g.built$panel$ranges[[1]]$x.range),
                               df.v$yvar/max(g.built$panel$ranges[[1]]$y.range),
                               df.v$yvar/min(g.built$panel$ranges[[1]]$y.range)))
    if(max(quantiles.of.limits)< .6){
      g <- g + geom_text(data = df.v,
                         aes(label = varname, x = xvar, y = yvar, angle = angle, hjust = hjust),
                         color = "darkred",
                         size = varname.size)
    }else{
      g <- g + geom_text(data = df.v,
                         aes(label = varname, x = xvar, y = yvar, angle = angle, hjust = "inward",vjust = "inward"),
                         color = "darkred",
                         size = varname.size)
    }
    

    

  }
  return(g)
}


#' Simple scatter plot
#' @param dat An nxp matrix of the expression data
#' @param probe.id a character string
#' @param probe.name a character string designating the probe label
#' @param smp.annot a single column data.frame containing x variable in the scatter plot
#' @return a plot object of class SPD and ggplot

scatter.by.probe <- function(dat, probe.id, probe.name = NULL, smp.annot){
  require(ggplot2)
  
  if(!is.matrix(dat))
    stop("dat needs to be a matrix")
  
  if(!probe.id %in% colnames(dat))
    stop("probe.id needs to be in colnames(dat)")
  
  if(is.null(probe.name))
    probe.name <- probe.id
  
  if(!is.data.frame(smp.annot))
    stop("smp.annot needs to be data.frame")
  if(ncol(smp.annot)!=1)
    stop("smp.annot needs to have only one column")
  smp.annot.name <- colnames(smp.annot)
  smp.annot <- smp.annot[,1]
    
  if(!is.numeric(smp.annot))
    stop("smp.annot needs to be numeric")
  
  df <- data.frame(smp.annot = smp.annot, y=dat[,probe.id])
  
  g <- ggplot(aes(y = y, x = smp.annot), data = df) +
    stat_smooth(method = "lm",se=T, colour="black",linetype ="dashed")+
    geom_point(colour = "chartreuse3",size =4) + 
    ggtitle(paste("Expression of",probe.name)) + 
    xlab(smp.annot.name) + ylab("Expression level") + 
    theme_bw()
  
  #cast class
  #----------
  class(g) <- c("SPD",class(g))
  return(g)
}


#' Simple violin plot
#' @param dat An nxp matrix of the expression data
#' @param probe.id a character string
#' @param probe.name a character string designating the probe label
#' @param smp.annot a single column data.frame containing x variable in the scatter plot
#' @return a plot object of class SPD and ggplot


violin.by.probe <- function(dat, probe.id, probe.name = NULL, smp.annot = NULL){
  require(ggplot2)
  
  if(!is.matrix(dat))
    stop("dat needs to be a matrix")
  
  if(!probe.id %in% colnames(dat))
    stop("probe.id needs to be in colnames(dat)")
  
  if(is.null(probe.name))
    probe.name <- probe.id
  
  if(is.null(smp.annot))
    smp.annot <- as.factor(rep("All observations",nrow(dat)))
  
  
  if(!is.data.frame(smp.annot))
    stop("smp.annot needs to be data.frame")
  if(ncol(smp.annot)!=1)
    stop("smp.annot needs to have only one column")
  smp.annot.name <- colnames(smp.annot)
  smp.annot <- smp.annot[,1]
  
  
  if(is.numeric(smp.annot))
    smp.annot <- factor(paste("Group",smp.annot))
  
  df <- data.frame(smp.annot = smp.annot, y=dat[,probe.id])
  
  g <- ggplot(aes(y = y, x = smp.annot), data = df) + 
    geom_violin(fill = "grey90", color = "white") +  
    geom_point(colour = "chartreuse3") + 
    ggtitle(paste("Expression of",probe.name)) +xlab(NULL)+ ylab("Expression level") + 
    geom_boxplot(width = .2,alpha=0)+
    theme_bw()
  if(nlevels(smp.annot)>1)
    g <- g + xlab(smp.annot.name)
  #theme(axis.text.x  = element_text(angle=60, vjust=0.5, size=10))
  #cast class
  #----------
  class(g) <- c("SPD",class(g))
  return(g)
}



#' This funciton generates PCA biplots using ggbiplot
#' @param dat An nxp matrix of the expression data
#' @param probe.ids A character vector of probe ids
#' @param probe.names A 1D data.frame with rownames as probe.ids and a column probe.label (column name is ignored)
#' @param probe.types A 1D data.frame with rownames as probe.ids and a column analyte (column name is ignored)
#' @param smp.annot A single column data.frame containing sample annoation. Note, the variable is expected to be categorical and if not it will be binned to be so
#' @return A plot object of class SPD, pcabiplot and ggplot
 
pca.biplot <- function(dat,probe.ids = NULL, probe.names = NULL, probe.types = NULL, smp.annot = NULL){
  
  if(!is.matrix(dat))
    stop("dat needs to be a matrix")
  
  if(is.null(probe.ids))
    probe.ids <- colnames(dat)
  
  if(is.null(probe.names))
    probe.names <- data.frame(row.names = probe.ids, probe.label = probe.ids)
  
  if(is.null(probe.types))
    probe.types <- data.frame(row.names = probe.ids, analyte = probe.ids)
  
  
  if(any(!probe.ids %in% colnames(dat)))
    stop("all probe.ids needs to be in colnames(dat)")
  
  if(!is.data.frame(probe.names))
    stop("probe.names is not a data.frame")
  if(any(!probe.ids %in% rownames(probe.names)))
    stop("rownames of probe.names don't include probe.ids to be described")
  if(ncol(probe.names)!=1)
    stop(paste("probe.names has",ncol(probe.names),". It shoudl have 1 column!"))
  
  if(!is.data.frame(probe.types))
    stop("probe.types is not a data.frame")
  if(any(!probe.ids %in% rownames(probe.types)))
    stop("rownames of probe.types don't include probe.ids to be described")
  if(ncol(probe.types)!=1)
    stop(paste("probe.types has",ncol(probe.names),". It shoudl have 1 column!"))
  
  prb.labs <- paste(probe.types[probe.ids,1],probe.names[probe.ids,1],sep="_")
  names(prb.labs) <- probe.ids
  dat <- dat[,probe.ids]
  colnames(dat) <- prb.labs[colnames(dat)]
  pcobj <- prcomp(x = dat,scale. = T,center = T)
  
  #If stratifying variable provided confirm/trichotomize as categorical
  #---------------------------------------------------------------------
  if(is.numeric(smp.annot[,1])){
    tmp <- rep("Low",nrow(smp.annot))
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .25,na.rm = T)]<- "Average"
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .75,na.rm = T)]<- "High"
    tmp[is.na(smp.annot[,1])] <- NA
    smp.annot[,1] <- factor(tmp,levels = c("Low","Average","High"))
  }
  
  
  g1 <- ggbiplot(pcobj = pcobj,choices = 1:2,groups = smp.annot[,1],ellipse = all(table(smp.annot)>3),circle = F,alpha = 1,var.axes = T,var.scale = 1)
  g1 <- g1 + ggtitle("PCA: samples on 1st and 2nd PC plane")
  g1 <- g1 + theme_bw()
  
  g2 <- ggbiplot(pcobj = pcobj,choices = c(1,3),groups = smp.annot[,1],ellipse = all(table(smp.annot)>3),circle = F,alpha = 1,var.axes = T,var.scale = 1)
  g2 <- g2 + ggtitle("PCA: samples on 1st and 3rd PC plane")
  g2 <- g2 + theme_bw()
  
  g3 <- ggbiplot(pcobj = pcobj,choices = c(2,3),groups = smp.annot[,1],ellipse = all(table(smp.annot)>3),circle = F,alpha = 1,var.axes = T,var.scale = 1)
  g3 <- g3 + ggtitle("PCA: samples on 2nd and 3rd PC plane")
  g3 <- g3 + theme_bw()
  
  # Change the title of the legend
  #-------------------------------
  
  g1 <- g1 + labs(color=colnames(smp.annot))
  g2 <- g2 + labs(color=colnames(smp.annot))
  g3 <- g3 + labs(color=colnames(smp.annot))
  
  #cast class
  #----------
  class(g1) <- class(g2) <- class(g3) <- c("pcabiplot","SPD",class(g1))
  
  return(list(pc12=g1,pc13=g2,pc23=g3))
  
}


#' This funciton generates parallel coordinate plots as well as the average trend
#' @param dat An nxp matrix of the expression data
#' @param probe.ids A character vector of probe ids
#' @param probe.names A 1D data.frame with rownames as probe.ids and a column probe.label (column name is ignored)
#' @param probe.types A 1D data.frame with rownames as probe.ids and a column analyte (column name is ignored)
#' @param smp.annot A single column data.frame containing sample annoation. Note, the variable is expected to be categorical and if not it will be binned to be so
#' @param scale.by.probe A logical when TRUE, all probes expression will be z-transformed prior to plotting
#' @return A plot object of class SPD, pcabiplot and ggplot

parcor.plot <- function(dat, 
                        probe.ids = NULL,
                        probe.names = NULL,
                        probe.types = NULL,
                        smp.annot,
                        scale.by.probe = T){
  require(reshape)
  require(ggplot2)
  
  if(!is.matrix(dat))
    stop("dat needs to be a matrix")
  
  if(is.null(probe.ids))
    probe.ids <- colnames(dat)
  
  if(is.null(probe.names))
    probe.names <- data.frame(row.names = probe.ids, probe.label = probe.ids)
  
  if(is.null(probe.types))
    probe.types <- data.frame(row.names = probe.ids, analyte = probe.ids)
  
  
  
  if(any(!probe.ids %in% colnames(dat)))
    stop("all probe.ids needs to be in colnames(dat)")
  
  if(!is.data.frame(probe.names))
    stop("probe.names is not a data.frame")
  if(any(!probe.ids %in% rownames(probe.names)))
    stop("rownames of probe.names don't include probe.ids to be described")
  if(ncol(probe.names)!=1)
    stop(paste("probe.names has",ncol(probe.names),". It shoudl have 1 column!"))
  
  if(!is.data.frame(probe.types))
    stop("probe.types is not a data.frame")
  if(any(!probe.ids %in% rownames(probe.types)))
    stop("rownames of probe.types don't include probe.ids to be described")
  if(ncol(probe.types)!=1)
    stop(paste("probe.types has",ncol(probe.names),". It shoudl have 1 column!"))
  
  
  if(length(unique(smp.annot[,1]))<2)
    stop("There should be at least two smaple categories in smp.annot")
  
  mat <- dat[,probe.ids]
  if(scale.by.probe)
    mat <- scale(mat)
  
  
  #If stratifying variable provided confirm/trichotomize as categorical
  #---------------------------------------------------------------------
  if(is.numeric(smp.annot[,1])){
    tmp <- rep("Low",nrow(smp.annot))
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .25,na.rm = T)]<- "Average"
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .75,na.rm = T)]<- "High"
    tmp[is.na(smp.annot[,1])] <- NA
    smp.annot[,1] <- factor(tmp,levels = c("Low","Average","High"))
  }
  
  ## if scale by group
  # for(grp in levels(groups)){
  #   mat[groups==grp,] <- scale(mat[groups==grp,])
  # }
  
  
  groups <- smp.annot[,1]
  if(is.numeric(groups))
    groups <- factor(paste("Group",smp.annot))
  
  # Make the dataframe and convert to long
  df <- data.frame(Sample.ID=rownames(mat),groups = groups, mat,check.names = F)
  long.dat <- melt(df, id.vars  = c("Sample.ID","groups"))
  tmp <- colnames(long.dat);tmp[tmp=="variable"] <- "probe"; tmp[tmp=="value"]<- "Expression"
  colnames(long.dat) <- tmp
  
  # Make and add order (Time)
  ord <- as.numeric(factor(long.dat$probe))
  long.dat$ord <- ord
  
  #Add analyte
  long.dat$probe.type <- probe.types[as.character(long.dat$probe),]
  
  # Get labels
  xlabel <- paste(probe.types[levels(long.dat$probe),], probe.names[levels(long.dat$probe),],sep="\n")
  
  # Generate the ggplot object
  p <-  ggplot(data = long.dat, aes(x = ord, y = Expression, group = Sample.ID)) 
  p <- p + geom_line(alpha=0.3,aes(colour = groups))
  #   p <- p + stat_smooth(aes(group = groups))
  p <- p + stat_summary(aes(group = groups,colour=groups),geom = "line",linetype = 2,fun.y = mean, size = 2)
  p <- p + stat_summary(aes(group = groups,shape = probe.type,colour = groups),geom = "point", fun.y = mean, size = 5)
  #   p <- p + facet_grid(groups ~ .) 
  p <- p + scale_x_discrete(name="", limits= xlabel)
  p <- p + ggtitle(paste("Parallel coordinate plot by",colnames(smp.annot)))
  p <- p + theme(axis.text.x  = element_text(angle=60, vjust=0.5, size=15))
  p <- p + theme_bw()
  
  #cast class
  #----------
  class(p) <- c("parcorplot","SPD",class(p))
  return(p)
}




#' Similarity plots:
#' Given a set of probe ids, it demonstrats their inter-relation. If only one probe is provided, it finds top probes that co-vary 
#' with the probe of interest. Specifically it
#' 1- Generates density plot for the probes of interest
#' 2- Generates pair-wise scatter plot among these probes and report the pearson cor as well as the (BIC choice of linear or non-linear) association
#' 3- Plots the (BIC choice of linear or non-linear) fit for each scatter plot 
#' 4- If sample annotation is provided, it reports the correlations and colorcode the plots by covariate 
#' @param dat An nxp normalized data matrix
#' @param probe.ids a character vector containing ids of the probes of interest
#' @param probe.labels optional labels to be used for display, default is probe.ids. It is named character vector of the same length and names as probe.ids
#' @param probe.types optional analyte type to added to the labels
#' @param smp.annot optioanl nx1 data.frame contraining only the column of the categorical sample annotation of interest for stratification. Note, the variable is expected to be categorical and if not it will be binned to be so
#' @param num.pairs.to.compare number of pairs of probes to compare the probe of intrest against. 
#' @param test.pairwise.assoc: logical. If TRUE, it finds the best linear or non-linear fit, plots and reports the p-value
#' @return A grpaical object of type corplot, SPD ggally and ggplot

corplot <- function(dat,
                    probe.ids,
                    probe.names = NULL,
                    probe.types = NULL,
                    smp.annot = NULL,
                    num.pairs.to.compare = 5,
                    test.pairwise.assoc = TRUE){
  require(ggplot2)
  require(GGally)
  require(mgcv)
  
  # custom function for association plots between prb.x and prb.y
  # fits bast (del BIC>2) of linear or non-linear fit 
  # generates the ggplot object and reports the p-value for overall 
  # (linear or non-linear) association. It also reports the overall cor between prb.x and prb.y
  #--------------------------------------------------------------------------------------------
  get.association.plot <- function(df,prb.x,prb.y,group.col){
    
    # convert names so formula for the model fit won't fail
    #-----------------------------------------------------
    colnames(df) <- make.names(colnames(df))
    prb.x <- make.names(prb.x)
    prb.y <- make.names(prb.y)
    
    group.term <- paste(group.col,"+")
    if(nlevels(df[,group.col])==1){
      group.term <- NULL
      group.col = NULL
    }
    
    
    frm.lin <- as.formula(paste(prb.y,"~",group.term,prb.x))
    frm.nl <- as.formula(paste(prb.y,"~",group.term,"s(",prb.x,",k = 3, fx = T)"))
    
    mod.lin <- lm(formula = frm.lin,data = df)
    mod.nl <- gam(formula = frm.nl,data =df)
    
    g <- ggplot(data = df,mapping = aes_string(x = prb.x,y = prb.y,group = group.col)) + geom_point(aes_string(colour=group.col))
    if( BIC(mod.lin)  < (BIC(mod.nl)+2)  ){
      g <- g + stat_smooth(method = lm, aes_string(colour=group.col)) + theme_bw()
      assoc.pval <- as.matrix(anova(mod.lin))[prb.x,"Pr(>F)"]
      return(list(g=g,assoc.pval=assoc.pval,fit.type = "linear",cor.xy = cor(df[,c(prb.x,prb.y)])[1,2]))
    }
    
    g <- g + stat_smooth(data=df,method = "gam",formula = y~s(x,k = 3,fx = T),aes_string(colour=group.col)) + theme_bw()  + theme(legend.position='none')
    assoc.pval <- anova(mod.nl)$s.pv
    test.of.non.linearity.pval <- as.matrix(anova(mod.lin,mod.nl))[2,"Pr(>F)"]
    return(list(g=g,
                assoc.pval=assoc.pval,
                fit.type = "cubic spline",
                test.of.non.linearity.pval =test.of.non.linearity.pval,
                cor.xy = cor(df[,c(prb.x,prb.y)])[1,2]))
  }
  
  # verify data matrix
  #-------------------
  if(!is.matrix(dat) & !is.data.frame(dat))
    stop("dat needs to be a matrix or data.frame")
  dat <- as.matrix(dat)
  
  if(any(!sapply(data.frame(dat),class)%in% c("numeric","integer")))
    stop("dat needs to be numberic or integer")
  
  # verify probe annoations provided
  #---------------------------------
  if(any(!probe.ids %in% colnames(dat)))
    stop(paste(paste(probe.ids),"not in colnames dat"))
  
  if(is.null(probe.names))
    probe.names <- data.frame(probe.ids,row.names = probe.ids)
  
  if(!is.data.frame(probe.names))
    stop("probe.names needs to be a data.frame")
  
  if(ncol(probe.names)!=1)
    stop("probe.names should have only 1 column")
  
  
  if(any(!probe.ids %in% rownames(probe.names)))
    stop("probe.ids need to be in names of probe.names")
  
  
  if(is.null(probe.types))
    probe.types <- data.frame(probe.type = rep("typeless",length(probe.ids)),row.names =  probe.ids)
  
  if(!is.data.frame(probe.types))
    stop("probe.types needs to be data.frame")
  
  if(ncol(probe.types)!=1)
    stop("probe.types needs to have only 1 column")
  
  
  if(any(!probe.ids %in% rownames(probe.types)))
    stop("probe.ids need to be in names of probe.types")
  
  
  
  # verify smp.annot
  #-----------------
  
  if(is.null(smp.annot))
    smp.annot <- data.frame(smp.annot = as.factor(rep("all obs",nrow(dat))),row.names = rownames(dat) )
  
  if(!is.data.frame(smp.annot))
    stop("smp.annot is not a data.frame")
  
  if(ncol(smp.annot)!=1)
    stop("smp.annot needs to have only 1 column")
  
  if(nrow(smp.annot)!=nrow(dat))
    stop("nrow(smp.annot) does not match nrow(dat)")
  
  if(!all(rownames(smp.annot) == rownames(dat)))
    stop("rownames of smp.annot does not match rownames of dat")
  
  
  #If stratifying variable provided confirm/trichotomize as categorical
  #---------------------------------------------------------------------
  if(is.numeric(smp.annot[,1])){
    tmp <- rep("Low",nrow(smp.annot))
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .25,na.rm = T)]<- "Average"
    tmp[smp.annot[,1]> quantile(smp.annot[,1],probs = .75,na.rm = T)]<- "High"
    tmp[is.na(smp.annot[,1])] <- NA
    smp.annot[,1] <- factor(tmp,levels = c("Low","Average","High"))
  }
  
  
  #convert smp.annot,probe.names and probe.types data.frame to the relevant column
  #-------------------------------------------------------------------------------
  annot.name <- colnames(smp.annot)
  smp.annot <- factor(abbreviate(smp.annot[,1,drop=T],minlength = 9),levels = abbreviate(levels(smp.annot[,1,drop=T]),minlength = 9))
  
  # probe.names <- probe.names[probe.ids,1,drop=T]
  # probe.types <- probe.types[probe.ids,1,drop=T]
  
  if(is.numeric(smp.annot[annot.name]))
    smp.annot <- factor(paste("Group",smp.annot))
  
  if(!is.factor(smp.annot))
    stop("smp.annot needs to be factor of length nrow(dat)")
  
  
  #ttl <- paste("Expression distribution and association of",paste(unique(probe.names),collapse = ","))
  ttl <- paste("Pairwise expression associations")
  if(length(probe.ids)==1){
    cor.mat <- cor(dat) 
    top.correlated.probes <- colnames(cor.mat)[order(abs(cor.mat[probe.ids,]),decreasing = T)[1 + 1:min(num.pairs.to.compare,ncol(dat))]]
    probes.to.compare.to <- top.correlated.probes
    ttl <- paste("Pairwise expression association of",probe.names[probe.ids,1,drop=T],"and its top",min(num.pairs.to.compare,ncol(dat)),"correlated probes")
    probe.ids <- c(probe.ids,top.correlated.probes)

  }
  probe.names <- probe.names[probe.ids,1,drop=T]
  probe.types <- probe.types[probe.ids,1,drop=T]
  
  df<- data.frame(smp.annot = smp.annot,dat[,probe.ids],check.names = F)
  
  #Plot the grid with diagonal element filled in only
  #--------------------------------------------------
  columnLabels <- as.character(probe.names)
  if(length(unique(probe.types))>1)
    columnLabels <- paste(columnLabels,"\n",probe.types)
  
  group.col <- "smp.annot"
  if(nlevels(as.factor(df$smp.annot))>1){
    if(test.pairwise.assoc){
      gp <- ggpairs(data = data.frame(df),columns = 2:ncol(df),mapping = aes_string(colour = group.col),lower = "blank",title = paste(ttl,"color-coded by",annot.name),legends = T,columnLabels = columnLabels)
    }else{
      gp <- ggpairs(data = data.frame(df),columns = 2:ncol(df),mapping = aes_string(colour = group.col),title = paste(ttl,"color-coded by",annot.name),legends = T,columnLabels = columnLabels)
    }
    
  }else{
    gp <- ggpairs(data = data.frame(df),columns = 2:ncol(df),lower = "blank",title = ttl,legends = T, columnLabels=columnLabels)
  }
  
  
  # Fill the off diagonals with custom fit and stat on the fit
  #-----------------------------------------------------------
  if(test.pairwise.assoc){
    prbs.left <- probe.ids
    for(prb.x in probe.ids){
      i <- which(probe.ids == prb.x)
      prbs.left <- prbs.left[prbs.left !=prb.x]
      if(length(prbs.left)>0){
        for(prb.y in prbs.left){
          j <- which(probe.ids == prb.y)
          tmp <- get.association.plot(df = df,prb.x = prb.x,prb.y = prb.y,group.col = group.col)
          #txt <- paste("Association p val = ",format(tmp$assoc.pval,digits=2),
          #            "\n Fit:",tmp$fit.type)
          txt <- paste("p val = ",format(tmp$assoc.pval,digits=2))
          #       if(tmp$fit.type =="cubic spline")
          #         txt <- paste(txt,"\nTest of non-linearity p val = ", format(tmp$test.of.non.linearity.pval,digits=2))
          
          #get the cor obj
          upper <- ggplot_build(getPlot(gp,i,j))
          upper.y.range <- upper$panel$ranges[[1]]$y.range
          upper.x.range <- upper$panel$ranges[[1]]$x.range
          
          gp <- putPlot(gp,value = tmp$g,i = j,j = i)
          
          g.upper <- ggally_text(label = paste("Corr=", signif(tmp$cor.xy,3),"\n",txt,sep=""),xP = .5,yP = .9,xrange = upper.x.range,yrange = upper.y.range,aes(colour="gray45"))
          
          # Add group-wise stats if more the 1 group
          if(nlevels(df[,group.col])>1)
            g.upper <- g.upper + geom_text(data = upper$data[[2]], mapping = aes(x = x, y = y,label = label,color=label), hjust = .5, vjust = .5)
          
          gp <- putPlot(gp,g.upper,i = i,j = j)
        }
      }
      
    }
  }
  
  for(d in 1:(ncol(df)-1)){
    gp <- putPlot(gp,value = getPlot(gp,d,d) + theme_bw(),d,d)
  }
  
  gp <- gp + theme(legend.position = "none", 
                   panel.grid.major = element_blank(), 
                   axis.ticks = element_blank(), 
                   panel.border = element_rect( colour = "gray", fill = NA))
  
  class(gp) <- c("corplot","SPD",class(gp))
  
  return(gp)
}





#' This function generates trend plot with a verity options
#' @param dat A nxp dataframe or matrix containing the normalized data for probes of interest.
#' @param probe.ids a character vector containing ids of the probes of interest
#' @param probe.labels optional labels to be used for display, default is probe.ids. It is named character vector of the same length and names as probe.ids
#' @param probe.types optional analyte type to added to the labels
#' @param smp.annots A data.frame with matching rownames as the rownames of the dat and at least all the "var" columns below
#' @param var.to.trend.by name of the variable in smp.annots showing the observation unit id (e.g. patient or cellline id). 
#' @param var.to.order.obs.by name of the variable in smp.annots which shows the direction of progression in observations (e.g. time). It can be continuous or categorical
#' @param var.to.stratify.by name of the variable in dat by which we could stratify the observations. Note, a given level of var.to.trend.by should remain at 
#' @param adj.to.tmin: T/F subtract tmin or not                   

trend.plot <- function(dat,
                       probe.ids,
                       probe.names = NULL,
                       probe.types = NULL,
                       smp.annots,  # <- all of sample annotation nxk
                       var.to.trend.by,
                       var.to.order.obs.by,
                       var.to.stratify.by = NULL,
                       adj.to.tmin = T,
                       xlab = NULL){
  require(ggplot2)
  
  if(!is.matrix(dat) & !is.data.frame(dat))
    stop("dat needs to be a matrix or data.frame")
  dat <- as.matrix(dat)
  
  if(any(!sapply(data.frame(dat),class)%in% c("numeric","integer")))
    stop("dat needs to be numberic or integer")
  
  if(any(!probe.ids %in% colnames(dat)))
    stop(paste(paste(probe.ids),"not in colnames dat"))
  
  # verify smp.annots
  #-----------------
  if(!is.data.frame(smp.annots))
    stop("smp.annots is not a data.frame")
  
  if(nrow(smp.annots)!=nrow(dat))
    stop("nrow(smp.annots) does not match nrow(dat)")
  
  if(!all(rownames(smp.annots) == rownames(dat)))
    stop("rownames of smp.annots does not match rownames of dat")
  
  if(!var.to.trend.by %in% colnames(smp.annots))
    stop(paste(var.to.trend.by,"is not in colnames(samp.annot)"))
  
  if(! var.to.order.obs.by %in% colnames(smp.annots))
    stop(paste(var.to.order.obs.by,"is notin the colnames(smp.annots"))
  
  if(isTRUE(var.to.stratify.by == "no_annotation"))
    var.to.stratify.by <- NULL
  
  if(!is.null(var.to.stratify.by)){
    if(!var.to.stratify.by %in% colnames(smp.annots))
      stop(paste(var.to.stratify.by,"is not in colnames(smp.annots)"))
  }else{
    smp.annots <- data.frame(smp.annots,stratifying.group = "all obs",stringsAsFactors = T)
    var.to.stratify.by <- "stratifying.group"
  }
  
  
  if(!class(smp.annots[,var.to.order.obs.by]) %in% c("numeric","integer","factor"))
    stop(paste(var.to.order.obs.by,"is not numeric, integer, or factor"))
  
  
  # verify probe annotations provided
  #----------------------------------
  if(any(!probe.ids %in% colnames(dat) ))
    stop(paste(paste(probe.ids[!probe.ids %in% colnames(dat)]),"not in colnames(dat)"))
  
  if(is.null(probe.names))
    probe.names <- data.frame(row.names = probe.ids,probe.label = probe.ids)
  
  if(!is.data.frame(probe.names))
    stop("probe.names need to be a data.frame")
  
  if(any(!probe.ids %in% rownames(probe.names)))
    stop("probe.ids need to be in names of probe.names")
  
  if(is.null(probe.types))
    probe.types <- data.frame(row.names = probe.ids,analyte = "typeless")
  
  if(!is.data.frame(probe.types))
    stop("probe.type need to be a data.frame")
  
  
  if(any(!probe.ids %in% rownames(probe.types)))
    stop("probe.ids need to be in names of probe.types")
  
  
  
  
  # Merge and limit to complete cases
  #----------------------------------
  dat <- data.frame(smp.annots[,c(var.to.trend.by, var.to.order.obs.by, var.to.stratify.by),drop=F],dat[,probe.ids,drop=F],check.names = F)
  dat <- dat[complete.cases(dat),,drop=F]
  
  # Assign labels
  annot.labs <- c("Trend.ID", "Observation.Order", "Stratifying.Group")
  names(annot.labs) <- c(var.to.trend.by, var.to.order.obs.by, var.to.stratify.by)
  
  # relable for manipulation convenience
  colnames(dat)[1:3] <- annot.labs
  
  # cast stratifying variable as factor
  dat$Stratifying.Group <- as.factor(dat$Stratifying.Group)
  
  xaxis.vals <- dat$Observation.Order
  xaxis.is.numeric <- is.numeric(xaxis.vals)
  
  # This is to accomodate and label character/factor for x-axis in trend plots
  get.xaxis.labs <- function(x){return(x)}
  if(is.factor(xaxis.vals)){
    #tmp <- as.numeric(sapply(as.list(as.character(t.obs)), function (k)  strsplit(k, "[^[:digit:]]")[[1]][strsplit(k, "[^[:digit:]]")[[1]]!=""]
    xaxis.labs <- levels(as.factor(xaxis.vals))
    #xaxis.labs <- as.numeric(factor(xaxis.vals))
    get.xaxis.labs <- function(x){return(xaxis.labs[x])}
  }
  
  #   groups <- smp.subset
  #   if(is.numeric(groups))
  #     groups <- factor(paste("Group",smp.subset))
  
  #cast as factor (any other class hasn't been covered)
  #   smp.subset <- as.factor(smp.subset)
  
  #Get all levels of t before subsetting
  xlevels <- levels(factor(xaxis.vals))
  
  #   # Make the dataframe
  #   df <- data.frame(Trend.ID= trend.ids, t= t.obs, groups = groups, mat)
  #   colnames(df)[colnames(df)==gene] <- "Expression"
  
  # check no repeated sample given Trend.ID,time point and group
  dat$Trend.ID <- as.character(dat$Trend.ID)
  repeated <- any(duplicated(interaction(dat$Trend.ID,dat$Observation.Order,dat$Stratifying.Group)))
  dat$Trend.ID <- as.factor(dat$Trend.ID)
  
  #   #collapse by sample ID if repeated
  #   dat0 <- dat
  #   if(repeated & adj.to.tmin){
  #     frm <- as.formula(paste(paste(get("probe.ids"),sep="+"),"~",paste(var.to.trend.by,var.to.order.obs.by,var.to.stratify.obs.by,sep=":")))
  #     df <- aggregate(formula = frm, data = dat, FUN = mean)
  #   }
  #     
  
  
  # check each Trend.ID.i:groups has the same t_min
  all.same.tmin <- length(unique(aggregate(formula = Observation.Order ~ Trend.ID:Stratifying.Group, data = dat, FUN = function(x) min(as.numeric(factor(x = x,levels = xlevels))))$Observation.Order)) == 1
  
  
  #   if(!all.same.tmin)
  #     #unaggragate
  #     df <- df0
  
  # update adjust.to.tmin based on whether on all.same.tmin is true
  adj.to.tmin <- adj.to.tmin & all.same.tmin
  
  # Labeling
  xlab0 <- paste("Observations ordered by")
  if(is.null(xlab))
    xlab <- var.to.order.obs.by
  xlab <- paste(xlab0,xlab)
  
  
  
  ylab <- "Expression"
  if(adj.to.tmin)
    ylab <- paste("Expression (zeroed at 1st/initial",var.to.order.obs.by,")",sep="")
  
  
  dat0 <- dat
  # Adjust to t0
  if(adj.to.tmin){
    
    if(!xaxis.is.numeric)
      dat[,"Observation.Order"] <- as.numeric(factor(dat$Observation.Order,levels = xlevels))
    
    for(smp.i in levels(dat$Trend.ID)){
      for(gr in levels(dat$Stratifying.Group[dat$Trend.ID == smp.i])){
        rows.i <- dat$Trend.ID == smp.i & dat$Stratifying.Group == gr
        if(all(!rows.i))
          next
        if(!xaxis.is.numeric){
          # dat[rows.i,"Observation.Order"] <- as.numeric(factor(dat$Observation.Order[rows.i],levels = xlevels))
          dat[rows.i,probe.ids] <- sweep(dat[rows.i,probe.ids,drop=F],MARGIN = 2,STATS =  as.numeric(apply(dat[rows.i & dat$Observation.Order == 1,probe.ids,drop=F],2,mean)))
        }else{
          dat[rows.i,probe.ids] <- sweep(dat[rows.i,probe.ids,drop=F],MARGIN = 2,STATS =  as.numeric(apply(dat[rows.i & dat$Observation.Order == min(dat$Observation.Order),probe.ids,drop=F],2,mean)))
        }
      }
    }
  }
  
  
  
  # Generate the ggplot object
  
  stratification.lab <- NULL
  if(nlevels(dat$Stratifying.Group)>1)
    stratification.lab <- paste("stratified by",var.to.stratify.by)
  
  dfm <- reshape2:::melt.data.frame(data = dat,id.vars = annot.labs,variable.name =  "Probe.id",value.name = "Expression")
  dfm$Type <- as.factor(probe.types[as.character(dfm$Probe.id),])
  dfm$Probe <- as.character(paste(dfm$Type,probe.names[as.character(dfm$Probe.id),],sep="\n"))
  
  if(!repeated){
    p <-  ggplot(data = dfm, aes(x = Observation.Order, y = Expression, group = interaction(Trend.ID,Probe))) 
    p <- p + geom_line(mapping = aes(colour=Probe),alpha=0.5)
    p <- p + stat_smooth(method = "gam",formula = y~s(x,k=3),aes(group = Probe,colour =Probe,linetype = Type),size=1.5)
    p <- p + stat_summary(aes(group = Probe, shape = Probe,colour = Probe),geom = "point", fun.y = mean, size = 3)
    p <- p + facet_grid(. ~ Stratifying.Group) 
    p <- p + ggtitle(paste("Expression trend for",paste(unique(probe.names[probe.ids,]),collapse = ", "),stratification.lab)) + xlab(xlab) + ylab(ylab)
    p <- p + scale_x_continuous(label=get.xaxis.labs)
    p <- p + theme_bw()
    
    class(p) <- c("trendplot","SPD",class(p))
    return(p)
  }
  
  p <-  ggplot(data = dfm, aes(x = Observation.Order, y = Expression, group = interaction(Trend.ID,Probe))) 
  p <- p + geom_point(mapping = aes(colour=Probe),position = "jitter")
  p <- p + stat_smooth(method = "gam",formula = y~s(x,k=3),aes(group = Probe,colour =Probe,linetype = Type),size=1.5)
  #     p <- p + stat_summary(aes(group = Probe, shape = Probe,colour = Probe),geom = "point", fun.y = mean, size = 3)
  p <- p + facet_grid(. ~ Stratifying.Group) 
  p <- p + ggtitle(paste("Expression trend for",paste(unique(probe.names[probe.ids,]),collapse = ", "),stratification.lab,"(points jittered)")) + xlab(xlab) + ylab(ylab)
  p <- p + scale_x_continuous(label=get.xaxis.labs)
  p <- p + theme_bw()
  
  class(p) <- c("trendplot","SPD",class(p))
  return(p)
  
}



#' This function generates the likely network of conditional dependencies between seletec probes
#' @param dat A nxp dataframe or matrix containing the normalized data for probes of interest.
#' @param probe.ids a character vector containing ids of the probes of interest
#' @param probe.names optional labels to be used for display, default is probe.ids. It is named character vector of the same length and names as probe.ids
#' @param probe.types optional analyte type to added to the labels
#' @param smp.annots A data.frame with matching rownames as the rownames of the dat and at least all the "var" columns below
#' @param adj.for.var A varaible name corresponding to either a numeric or categorical variable to be used to adjust prior to network estimation
#' @param significance.alpha A numeric value. Significance threshold when F.test is used for edge selection test method
#' @param significance.delBIC A numeric value. Significance threshold when BIC is used for edge selection test method
#' @param test.method Either "BIC" or "F.test". The method to be used for edge selection

interactnet.plot <- function(dat,
                             probe.ids,
                             probe.names = NULL,
                             probe.types = NULL,
                             smp.annot,
                             adj.for.var = "no_annotation",
                             significance.alpha = .05,
                             significance.delBIC = 2,
                             test.method = c("BIC","F.test")){
  
  get.interactnet.adjMat <- function(dat,smp.annot,probe.ids,adj.for.var = NULL,significance.alpha,significance.delBIC,test.method){
    
    if(!is.null(adj.for.var)){
      if(!adj.for.var %in% colnames(smp.annot))
        stop("Variable by which to adjust InteractNet is not in the sample annoation")
    }
    num.adj.terms <- ifelse(is.null(adj.for.var),0,ifelse(is.numeric(smp.annot[,adj.for.var]),1,nlevels(smp.annot[,adj.for.var])-1 ))
    num.terms <- length(probe.ids[-1]) + 1 + num.adj.terms
    dof <- nrow(dat) - num.terms
    if(dof<1)
      return(NULL)
    
    #make probe.ids generic
    #----------------------
    names(probe.ids) <- paste("p",1:length(probe.ids),sep="")
    dat <- scale(dat[,probe.ids]); colnames(dat) <- names(probe.ids); dat <- data.frame(smp.annot,dat)
    
    node.names <- probe.ids
    adj.mat <-  significance.mat <- matrix(0,nrow = length(node.names), ncol= length(node.names), dimnames = list(node.names,node.names) )
    for(prb.y in names(probe.ids)){
      prbs.x <- names(probe.ids)[names(probe.ids) !=prb.y] 
      #       frm.i <- as.formula(paste(prb.y,"~",effect.modifier.var,"+",paste(prbs.x,collapse = "+")))
      frm.i <- as.formula(paste(prb.y,"~",paste(prbs.x,collapse = "+")))
      if(!is.null(adj.for.var))
        frm.i <- as.formula(paste(prb.y,"~",paste(c(prbs.x,adj.for.var),collapse = "+")))
      #       frm.i <- as.formula(paste(prb.y,"~",effect.modifier.var,"+",paste(prbs.x,collapse = "+"),"+",effect.modifier.var,":(",paste(prbs.x,collapse = "+"),")"))
      full.mod.i <- lm(formula = frm.i,data = dat)
      BIC.full <- BIC(full.mod.i)
      cf.i <- summary(full.mod.i)$coefficients
      cf.i <- cf.i[rownames(cf.i) !="(Intercept)",,drop=F]
      #rownames(cf.i) <- attr(attr(model.frame(frm.i,data=dat),"terms"),"term.labels")
      cf.i <- as.data.frame(cf.i)
      
      
      for(prb.x in prbs.x){
        prbs.x.drop <- prbs.x[prbs.x!=prb.x]
        frm.i.drop <- paste(prb.y, "~ 1")
        if(length(prbs.x.drop)>0)
          frm.i.drop <- paste(prb.y,"~",paste(prbs.x.drop,collapse = "+"))
        
        if(!is.null(adj.for.var))
          frm.i.drop <- paste(c(frm.i.drop,adj.for.var),collapse = "+")
        frm.i.drop <- as.formula(frm.i.drop)
        
        dropmod.prb.x <- lm(frm.i.drop,data=dat)
        BIC.dropmod.prbx <- BIC(dropmod.prb.x)
        
        if(test.method == "BIC"){
          significance.mat[probe.ids[prb.x],probe.ids[prb.y]] <- BIC.dropmod.prbx - BIC.full
          keep.edge <- significance.mat[probe.ids[prb.x],probe.ids[prb.y]]>significance.delBIC
          
        }
        
        if(test.method == "F.test"){
          significance.mat[probe.ids[prb.x],probe.ids[prb.y]] <- anova(full.mod.i,dropmod.prb.x)[2,"Pr(>F)"]
          keep.edge <- significance.mat[probe.ids[prb.x],probe.ids[prb.y]] < significance.alpha
          
        }
        
        adj.mat[probe.ids[prb.x],probe.ids[prb.y]] <- keep.edge *cf.i[prb.x,"Estimate"] 
      } 
    }
    
    if(test.method == "BIC")
      significance.mat <- pmax(significance.mat,0)
    if(test.method == "F.test"){
      significance.mat[significance.mat>significance.alpha] <- 1
      significance.mat <- -log10(significance.mat)
    }
    
    significance.mat[adj.mat==0] <- 0
    
    return(list(adj.mat=adj.mat, significance.mat = significance.mat))
    
  }
  
  
  
  if(length(probe.ids)<2)
    stop("interactNet plot requires at least 2 probes to be selected.")
  
  test.method <- match.arg(arg = test.method)
  if(adj.for.var == "no_annotation")
    adj.for.var <- NULL
  tmp <- get.interactnet.adjMat(dat = dat,  
                                smp.annot = smp.annot,
                                probe.ids = probe.ids,
                                adj.for.var =  adj.for.var,
                                significance.alpha =significance.alpha,
                                significance.delBIC = significance.delBIC,
                                test.method = test.method)
  
  if(is.null(tmp))
    return(NULL)
  
  adj.mat <- tmp$adj.mat
  adj.sig.mat <- tmp$significance.mat
  
  #average the effect
  adj.mat <- matrix(apply(cbind(as.numeric(adj.mat),as.numeric(t(adj.mat))),1,mean),nrow = nrow(adj.mat),ncol=ncol(adj.mat),dimnames = dimnames(adj.mat))
  
  require(igraph)
  require(grid)
  g <- graph.adjacency(adj.mat,mode = "undirected",weighted=TRUE)
  g.sig <- graph.adjacency(adj.sig.mat,mode = "undirected",weighted=TRUE)
  #   get.data.frame(g)
  # plot(g,layout=layout.circle)
  xy <- data.frame(scale(layout.circle(g)))
  dimnames(xy) <- list(rownames(adj.mat),c("x","y"))
  xy <- data.frame(label = NA,type=NA,xy)
  xy[rownames(xy) %in% probe.ids,"label"] <- as.character(probe.names[rownames(xy)[rownames(xy) %in% probe.ids],])
  xy[rownames(xy) %in% probe.ids,"type"] <- as.character(probe.types[rownames(xy)[rownames(xy) %in% probe.ids],])
  
  # if(effect.modifier.var %in% rownames(xy))
  #   xy[effect.modifier.var,c("label","type")] <- c(effect.modifier.var,"Effect modifier")
  
  # Set up edge
  
  edge.xy <- data.frame(x=xy[get.edgelist(g)[,1],"x"],y=xy[get.edgelist(g)[,1],"y"],xend = xy[get.edgelist(g)[,2],"x"],yend=xy[get.edgelist(g)[,2],"y"])
  arrow.gap <- .25
  x.length = with(edge.xy, xend - x)
  y.length = with(edge.xy, yend - y)
  arrow.gap = with(edge.xy, arrow.gap/sqrt(x.length^2 +  y.length^2))
  edge.xy = transform(edge.xy, 
                      x = x + arrow.gap * x.length, 
                      y = y + arrow.gap * y.length, 
                      xend = x + (1 - arrow.gap) * x.length,
                      yend = y + (1 - arrow.gap) * y.length)
  colnames(edge.xy) <- paste("e",colnames(edge.xy),sep=".")
  
  
  edge.xy <- data.frame(edge.xy,Effect = get.data.frame(g)$weight, Significance = get.data.frame(g.sig)$weight)
  
  ttl <- paste("Conditional dependencies between",paste(unique(probe.names[probe.ids,]),collapse = ","))
  if(!is.null(adj.for.var))
    ttl <- paste("Conditional dependencies between",paste(unique(probe.names[probe.ids,]),collapse = ","),"\n Adjusted for:",adj.for.var)
  if(test.method == "BIC")
    ttl <- paste(ttl,"\nEdge Selection via BIC (delta BIC >",significance.delBIC,")")
  
  if(test.method == "F.test")
    ttl <- paste(ttl,"\nEdge Selection via F test (alpha value <",significance.alpha,")")
  
  #change type ==> Type in colname
  if("type" %in% colnames(xy))
    colnames(xy)[colnames(xy)=="type"] <- "Type"
  
  
  gp <- ggplot(data = xy,mapping = aes(x = x,y = y)) 
  gp <- gp  + xlim(-2,2) + ylim(-2,2) + geom_point(size=20,mapping = aes(shape = Type),colour = "gray") +geom_text(data = xy,aes(x,y,label = label) )
  
  if(nrow(edge.xy)>0)
    gp <- gp + geom_segment(data = edge.xy, mapping = aes(x = e.x, y = e.y,xend = e.xend, yend = e.yend,colour=Effect,size=Significance))
  
  gp <- gp + theme(panel.background = element_blank(),
                   panel.grid = element_blank(), 
                   axis.title = element_blank(), 
                   legend.key = element_blank(),
                   axis.ticks = element_blank(),
                   axis.text.x = element_blank(),
                   axis.text.y = element_blank())
  gp <- gp + ggtitle(ttl)
  
  #cast class
  #----------
  class(gp) <- c("interactnet","SPD",class(gp))
  return(gp)
}



# interactnet.plot <- function(dat,
#                              probe.ids,
#                              probe.names = NULL,
#                              probe.types = NULL,
#                              smp.annot,
#                              effect.modifier.var = NULL,
#                              significance.alpha = .05,
#                              significance.delBIC = 2,
#                              test.method = c("BIC","F.test")){
#   
#   # This function gets adjacency matrix for a newtwork of interactions 
#   get.interactnet.adjMat <- function(dat,smp.annot,probe.ids,effect.modifier.var = NULL,significance.alpha,significance.delBIC,test.method){
#     dat <- data.frame(smp.annot,dat[,probe.ids])
#     node.names <- c(effect.modifier.var,probe.ids)
#     
#     adj.mat <-  significance.mat <- matrix(0,nrow = length(node.names), ncol= length(node.names), dimnames = list(node.names,node.names) )
#     for(prb.y in probe.ids){
#       prbs.x <- probe.ids[probe.ids !=prb.y] 
#       #       frm.i <- as.formula(paste(prb.y,"~",effect.modifier.var,"+",paste(prbs.x,collapse = "+")))
#       frm.i <- as.formula(paste(prb.y,"~",paste(prbs.x,collapse = "+")))
#       if(!is.null(effect.modifier.var))
#         frm.i <- as.formula(paste(prb.y,"~",effect.modifier.var,"+",paste(prbs.x,collapse = "+"),"+",effect.modifier.var,":(",paste(prbs.x,collapse = "+"),")"))
#       full.mod.i <- lm(formula = frm.i,data = dat)
#       BIC.full <- BIC(full.mod.i)
#       cf.i <- summary(full.mod.i)$coefficients
#       cf.i <- cf.i[rownames(cf.i) !="(Intercept)",]
#       rownames(cf.i) <- attr(attr(model.frame(frm.i,data=dat),"terms"),"term.labels")
#       cf.i <- as.data.frame(cf.i)
#       
#       
#       #       cf.i$edge <- cf.i$Estimate*as.numeric(cf.i$"Pr(>|t|)"< significance.alpha)
#       #       cf.i$log.p <- -log10(cf.i$"Pr(>|t|)"); cf.i[(cf.i$"Pr(>|t|)"> significance.alpha),"log.p"] <-0
#       
#       
#       
#       if(!is.null(effect.modifier.var)){
#         frm.drop.EM <- as.formula(paste(prb.y,"~",paste(prbs.x,collapse = "+")))
#         dropmod.EM <- lm(frm.drop.EM,data=dat)
#         BIC.dropmod.EM <- BIC(dropmod.EM)
#         
#         if(test.method == "BIC"){
#           significance.mat[effect.modifier.var,prb.y] <- BIC.dropmod.EM - BIC.full
#           keep.edge <- significance.mat[effect.modifier.var,prb.y]>significance.delBIC
#           
#         }
#           
#         if(test.method == "F.test"){
#           significance.mat[effect.modifier.var,prb.y] <- anova(full.mod.i,dropmod.EM)[2,"Pr(>F)"]
#           keep.edge <- significance.mat[effect.modifier.var,prb.y] < significance.alpha
#           
#         }
#           
#         adj.mat[effect.modifier.var,prb.y] <- as.numeric(keep.edge)*cf.i[effect.modifier.var,"Estimate"]
#       }
#       
# 
#       for(prb.x in prbs.x){
#         prbs.x.drop <- prbs.x[prbs.x!=prb.x]
#         
#         frm.i.drop <- as.formula(paste(prb.y,"~",paste(prbs.x.drop,collapse = "+")))
#         if(!is.null(effect.modifier.var))
#           frm.i.drop <- as.formula(paste(prb.y,"~",effect.modifier.var,"+",paste(prbs.x.drop,collapse = "+"),"+",effect.modifier.var,":(",paste(prbs.x.drop,collapse = "+"),")"))
#         
#         dropmod.prb.x <- lm(frm.i.drop,data=dat)
#         BIC.dropmod.prbx <- BIC(dropmod.prb.x)
#         
#         if(test.method == "BIC"){
#           significance.mat[prb.x,prb.y] <- BIC.dropmod.prbx - BIC.full
#           keep.edge <- significance.mat[prb.x,prb.y]>significance.delBIC
#           
#         }
#         
#         if(test.method == "F.test"){
#           significance.mat[prb.x,prb.y] <- anova(full.mod.i,dropmod.prb.x)[2,"Pr(>F)"]
#           keep.edge <- significance.mat[prb.x,prb.y] < significance.alpha
#           
#         }
#         
#         adj.mat[prb.x,prb.y] <- keep.edge *cf.i[prb.x,"Estimate"] 
#       }
# 
#     }
#     
#     if(test.method == "BIC")
#       significance.mat <- pmax(significance.mat,0)
#     if(test.method == "F.test"){
#       significance.mat[significance.mat>significance.alpha] <- 1
#       significance.mat <- -log10(significance.mat)
#     }
#     
#     significance.mat[adj.mat==0] <- 0
#     
#     return(list(adj.mat=adj.mat, significance.mat = significance.mat))
#   }
#   
#   test.method <- match.arg(arg = test.method)
#   tmp <- get.interactnet.adjMat(dat = dat, 
#                                 smp.annot = smp.annot,
#                                 probe.ids = probe.ids,
#                                 effect.modifier.var = effect.modifier.var,
#                                 significance.alpha =significance.alpha,
#                                 significance.delBIC = significance.delBIC,
#                                 test.method = test.method)
#                                 
#   adj.mat <- tmp$adj.mat
#   adj.sig.mat <- tmp$significance.mat
#   
#   require(igraph)
#   require(grid)
#   g <- graph.adjacency(adj.mat,weighted=TRUE)
#   g.sig <- graph.adjacency(adj.sig.mat,weighted=TRUE)
#   #   get.data.frame(g)
#   #   plot(g,layout=layout.star)
#   xy <- data.frame(scale(layout.star(g)))
#   dimnames(xy) <- list(rownames(adj.mat),c("x","y"))
#   xy <- data.frame(label = NA,type=NA,xy)
#   xy[rownames(xy) %in% probe.ids,"label"] <- as.character(probe.names[rownames(xy)[rownames(xy) %in% probe.ids],])
#   xy[rownames(xy) %in% probe.ids,"type"] <- as.character(probe.types[rownames(xy)[rownames(xy) %in% probe.ids],])
#   
#   if(effect.modifier.var %in% rownames(xy))
#     xy[effect.modifier.var,c("label","type")] <- c(effect.modifier.var,"Effect modifier")
#   
#   # Set up edge
# 
#   edge.xy <- data.frame(x=xy[get.edgelist(g)[,1],"x"],y=xy[get.edgelist(g)[,1],"y"],xend = xy[get.edgelist(g)[,2],"x"],yend=xy[get.edgelist(g)[,2],"y"])
#   arrow.gap <- .25
#   x.length = with(edge.xy, xend - x)
#   y.length = with(edge.xy, yend - y)
#   arrow.gap = with(edge.xy, arrow.gap/sqrt(x.length^2 +  y.length^2))
#   edge.xy = transform(edge.xy, 
#                       x = x + arrow.gap * x.length, 
#                       y = y + arrow.gap * y.length, 
#                       xend = x + (1 - arrow.gap) * x.length,
#                       yend = y + (1 - arrow.gap) * y.length)
#   colnames(edge.xy) <- paste("e",colnames(edge.xy),sep=".")
#   
#   
#   edge.xy <- data.frame(edge.xy,Effect = get.data.frame(g)$weight, significance = get.data.frame(g.sig)$weight)
#   
#   ttl <- paste("Conditional dependencies between",paste(unique(probe.names[probe.ids,]),collapse = ","))
#   if(!is.null(effect.modifier.var))
#     ttl <- paste("Conditional dependencies between",paste(unique(probe.names[probe.ids,]),collapse = ","),"\n Effect modifier selected:",effect.modifier.var)
#   if(test.method == "BIC")
#     ttl <- paste(ttl,"\nEdge Selection via BIC (delta BIC >",significance.delBIC,")")
#   
#   if(test.method == "F.test")
#     ttl <- paste(ttl,"\nEdge Selection via F test (alpha value <",significance.alpha,")")
#   
#   gp <- ggplot(data = xy,mapping = aes(x = x,y = y)) 
#   gp <- gp  + xlim(-2,2) + ylim(-2,2) + geom_point(size=20,mapping = aes(shape = type),colour = "gray") +geom_text(data = xy,aes(x,y,label = label) )
#   gp <- gp + geom_segment(data = edge.xy, mapping = aes(x = e.x, y = e.y,xend = e.xend, yend = e.yend,colour=Effect,size=significance), arrow = arrow(type = "closed",length = unit(20, "pt")))
#   gp <- gp + theme(panel.background = element_blank(),
#                    panel.grid = element_blank(), 
#                    axis.title = element_blank(), 
#                    legend.key = element_blank(),
#                    axis.ticks = element_blank(),
#                    axis.text.x = element_blank(),
#                    axis.text.y = element_blank())
#   gp <- gp + ggtitle(ttl)
#   
#   #cast class
#   #----------
#   class(gp) <- c("interactnet","SPD",class(gp))
#   return(gp)
# }

#==========================================
# Functions to manipulate the plot objects
#==========================================

# This functions evalutes inputs to ensure they are colours
# @x a character vector of colours
# @value: a named vector of T/F
areColors <- function(x) {
  sapply(x, function(X) {
    tryCatch(is.matrix(col2rgb(X)), 
             error = function(e) FALSE)
  })
}


# Recieves the a parcorplot and colours and changes the the colours according to order
# Note, 1 colour needs to be provided for each of the levels of group
# @p a parcorplot object
# @colors: a chracter vector of acceptable of colours of the same length as levels of groups in the parcorplot
# hexidecimal or colors() are acceptable

parcor.plot.change.color <- function(p,colors){
  if(is.null(colors)){
    message("No coloring scheme is defined. Default color is used for parallel coordinate plot")
    return(p)
  }
  
  if(any(!areColors(colors))){
    warning(paste(paste(colors[!areColors(colors)],collapse = ", "),"should be replaced with colors"))
    return(p)
  }
  
  if(any(!class(p) == c("parcorplot","SPD","gg","ggplot")))
    stop("class plot needs to be parcorplot, gg and ggplot")
  
  built <- ggplot_build(plot = p)
  color.lookup.df.index <- which(unlist(lapply(built$data, function(x) "colour" %in% colnames(x))))[1]
  num.colors <- length(unique(built$data[[color.lookup.df.index]]$colour))
  if(length(colors)!=num.colors)
    stop("Exactly",num.colors,"colors are expected")
  
  return(p+scale_color_manual(values = colors))
}



# Recieves the a pca.biplot and colors and changes the the colours according to order
# Note, 1 color needs to be provided for each of the levels of group
# @ps a pca.biplot object
# @colors: a chracter vector of acceptable of colours of the same length as levels of groups in the parcorplot
# hexidecimal or colors() are acceptable
pca.biplot.change.color <- function(p,colors){
  if(is.null(colors)){
    message("No coloring scheme is defined. Default color is used for PCA biplot")
    return(p)
  }
  
  if(any(!areColors(colors))){
    warning(paste(paste(colors[!areColors(colors)],collapse = ", "),"should be replaced with colors"))
    return(p)
  }
  
  if(any(!class(p) == c("pcabiplot","SPD","gg","ggplot")))
    stop("class plot needs to be pcabiplot, gg and ggplot")
  
  #   if(any(!unlist(lapply(tmp,function(x) class(x) == c("pcabiplot","gg","ggplot")))))
  #     stop("class plot nees to be pcabiplot, gg and ggplot")
  
  built <- ggplot_build(plot = p)
  # color.lookup.df.index <- which(unlist(lapply(built$data, function(x) "colour" %in% colnames(x))))[1]
  # num.colors <- length(unique(built$data[[color.lookup.df.index]]$colour))
  num.colors <- length(unique(built$data[[2]]$colour))
  if(length(colors)!=num.colors)
    stop(paste("Exactly",num.colors,"colors are expected"))
  return(p+scale_color_manual(values = colors))
  
}
# Recieves the a trend.plot and colors and changes the the colors according to order
# Note, 1 color needs to be provided for each of the levels of group
# @p a trend.plot object
# @colors: a chracter vector of acceptable of colors of the same length as the number of probes
# hexidecimal or colors() are acceptable

trend.plot.change.color <- function(p, colors){
  if(is.null(colors)){
    message("No coloring scheme is defined. Default color is used for trend plot")
    return(p)
  }
  
  if(any(!areColors(colors))){
    warning(paste(paste(colors[!areColors(colors)],collapse = ", "),"should be replaced with colors"))
    return(p)
  }
  
  if(any(!class(p) == c("trendplot","SPD","gg","ggplot")))
    stop("class plot nees to be trendplot, gg and ggplot")
  
  built <- ggplot_build(plot = p)
  num.colors <- length(unique(built$data[[2]]$colour))
  if(length(colors)!=num.colors)
    stop("Exactly",num.colors,"colors are expected")
  
  return(p+scale_color_manual(values = colors))
}

corplot.change.color <- function(p, colors){
  if(is.null(colors)){
    message("No coloring scheme is defined. Default color is used for corplot")
    return(p)
  }
    
  if(any(!areColors(colors))){
    warning(paste(paste(colors[!areColors(colors)],collapse = ", "),"should be replaced with colors"))
    return(p)
  }
    
  
  if(any(!class(p) == c("corplot","SPD","gg","ggmatrix")))
    stop("class plot nees to be corplot, gg and ggplot")
  
  grid.ncolrow <- sqrt(length(p$plots))
  
  for(i in 1:grid.ncolrow){
    for(j in 1:grid.ncolrow){
      p.i <- getPlot(x = p,i = i,j = j)
      built <- ggplot_build(plot = p.i)
      
      num.colors <- 0
      if(i==j){
        num.colors <- length(unique(built$data[[1]]$fill))
        if(length(colors)!=num.colors)
          stop("Exactly",num.colors,"colors are expected",sep=" ")
        p <- putPlot(x =p,value = p.i + scale_fill_manual(values = colors),i = i,j = j)
      }
        
      if(j>i){
        num.colors <- length(unique(built$data[[2]]$colour))
        if(length(colors)!=num.colors)
          stop("Exactly",num.colors,"colors are expected",sep=" ")
        p <- putPlot(x =p,value = p.i + scale_color_manual(values = colors),i = i,j = j)
      }
       
      if(i>j){
        num.colors <- length(unique(built$data[[1]]$colour))
        if(length(colors)!=num.colors)
          stop("Exactly",num.colors,"colors are expected",sep=" ")
        p <- putPlot(x =p,value = p.i + scale_color_manual(values = colors),i = i,j = j)
      }
        
      
      

 
    }
  }
  

  
  return(p)
}


#wrapper change color function
SPD.plot.chnage.color <- function(p,colors){
  if("corplot" %in% class(p))
    return(corplot.change.color(p = p,colors = colors))
  if("parcorplot" %in% class(p))
    return(parcor.plot.change.color(p = p,colors = colors))
  if("pcabiplot" %in% class(p))
    return(pca.biplot.change.color(p = p,colors = colors))
  stop("The plot object not supported for color change")
}


#get color levels
SPD.plot.get.color.levels <- function(p){
  if("corplot" %in% class(p))
    return(corplot.get.color.levels(p = p))
  if("parcorplot" %in% class(p))
    return(parcorplot.get.color.levels(p = p))
  if("pcabiplot" %in% class(p))
    return(pcabiplot.get.color.levels(p = p))
  stop("The plot object not supported for color change")
}


corplot.get.color.levels <- function(p){
  p.21 <- getPlot(x = p,i = 2,j = 1)
  return(levels(p.21$plot_env$df$smp.annot))
}

parcorplot.get.color.levels <- function(p){
  return(levels(p$plot_env$df$groups))
}

pcabiplot.get.color.levels <- function(p){
  return(levels(p$plot_env$groups))
}

#' This function is just a wrapper to change the default
#' print parameters of corplot to allow for larger margins
print.corplot <- function(x,...){
  GGally:::print.ggmatrix(x, 
                          leftWidthProportion = 0.4,
                          bottomHeightProportion = 0.4, 
                          spacingProportion = 0.03,
                          gridNewPage = TRUE, ...)
}


#----------


# Test cases
#-----------
if(F){
  load("S:/AMashadi-Hossein/Take2/2015_05_12/R_FILES/data_n_annot.RData")
  normalized.dat <- e
  gene <- colnames(normalized.dat)[1]
  smp.annot.informative <- rep("informative.cat0",nrow(normalized.dat)); smp.annot.informative[normalized.dat[,gene]>median(normalized.dat[,gene]) ]<- "informative.cat1"
  smp.annot <- smp.annot.informative
  set.seed(19)
  
  # create some dummy sample ids
  smp.ids <- rep(NA,nrow(normalized.dat))
  smp.ids[smp.annot == unique(smp.annot)[1]] <- sample(1:4,size = sum(smp.annot == unique(smp.annot)[1]),replace = T)
  smp.ids[smp.annot == unique(smp.annot)[2]] <- sample(5:8,size = sum(smp.annot == unique(smp.annot)[2]),replace = T)
  smp.ids <- paste("smp.",smp.ids,sep="")
  
  t.obs <- rep(NA,nrow(normalized.dat))
  for(smp in unique(smp.ids)){
    t.obs[smp.ids == smp] <- 1:length(t.obs[smp.ids == smp])
  }
  trend.sannot <- data.frame(smp.annot.informative,smp.ids,t.obs,row.names = rownames(normalized.dat))
  table(trend.sannot$smp.ids,trend.sannot$t.obs,trend.sannot$smp.annot.informative)
  
  
  # , ,  = informative.cat0
  # 
  # 
  #       1 2 3 4 5 6 7 8 9 10
  # smp.1 1 1 1 1 0 0 0 0 0  0
  # smp.2 1 1 1 1 1 1 1 0 0  0
  # smp.3 1 1 1 1 1 0 0 0 0  0
  # smp.4 1 1 1 1 1 1 1 0 0  0
  # smp.5 0 0 0 0 0 0 0 0 0  0
  # smp.6 0 0 0 0 0 0 0 0 0  0
  # smp.7 0 0 0 0 0 0 0 0 0  0
  # smp.8 0 0 0 0 0 0 0 0 0  0
  # 
  # , ,  = informative.cat1
  # 
  # 
  #       1 2 3 4 5 6 7 8 9 10
  # smp.1 0 0 0 0 0 0 0 0 0  0
  # smp.2 0 0 0 0 0 0 0 0 0  0
  # smp.3 0 0 0 0 0 0 0 0 0  0
  # smp.4 0 0 0 0 0 0 0 0 0  0
  # smp.5 1 1 1 0 0 0 0 0 0  0
  # smp.6 1 1 1 1 1 0 0 0 0  0
  # smp.7 1 1 1 1 1 1 1 1 1  1
  # smp.8 1 1 1 1 1 0 0 0 0  0
  
  
  # dat <- normalized.dat
  # smp.annot <- data.frame(useless = "useless",trend.sannot)
  # prb.annot <- c("Protein",rep("mRNA",ncol(dat)-1));names(prb.annot) <- colnames(normalized.dat)
  # probe.names <- colnames(normalized.dat)[1:2]
  # var.to.trend.by <- "smp.ids"
  # var.to.order.obs.by <- "t.obs"  
  # var.to.stratify.by = "smp.annot.informative"
  # adj.to.tmin = T 
  # xlab = NULL
  
  
  dat= normalized.dat
  colnames(dat) <- paste("probe", sprintf("%03d", 1:ncol(dat)),sep="")
  prb.annot <- data.frame(analyte=c("Protein",rep("mRNA",ncol(dat)-1)), 
                          probe.names=  colnames(normalized.dat),
                          row.names = colnames(dat))
  
  probe.names = prb.annot[,"probe.names",drop=F]
  probe.types <- prb.annot[,"analyte",drop=F]
  probe.ids <- rownames(prb.annot)[prb.annot$probe.names %in% c("CD1D", "TFRC","FPR2","CD24","CXCL5","MAP3K7","TNFSF12")]
  smp.annot <- data.frame(useless = "useless",trend.sannot)
  #   prb.annot <- c("Protein",rep("mRNA",ncol(dat)-1));names(prb.annot) <- colnames(normalized.dat)
  
  # col.annot = NULL,
  annot.name = "smp.annot.informative"
  num.pairs.to.compare = 5
  
  
  #==============
  # corplot test
  #==============
  corplot.by.probe(dat = dat,probe.names = probe.names,smp.annot = smp.annot,annot.name = annot.name)
  corplot.by.probe(dat = dat,
                   probe.ids = probe.ids,
                   probe.names = probe.names,
                   probe.types = probe.types,
                   smp.annot = smp.annot[,"smp.annot.informative",drop=F])
  #================
  #Trend plto test
  #================
  
  # repeated case
  #--------------
  
  # not all same tmin case
  #----------------------- 
                                                 
  
  var.to.trend.by <- "smp.ids"
  var.to.order.obs.by <- "t.obs"  
  var.to.stratify.by = "smp.annot.informative"
  adj.to.tmin = T 
  xlab = NULL
  
  trend.plot(dat = dat,smp.annot = smp.annot,prb.annot = prb.annot,probe.names = probe.names,var.to.trend.by = var.to.trend.by,var.to.order.obs.by = var.to.order.obs.by,var.to.stratify.by = var.to.stratify.by)
  trend.plot(dat = dat,
             probe.ids = probe.ids,
             probe.names = probe.names,
             probe.types = probe.types,
             smp.annot = smp.annot,
             var.to.trend.by = var.to.trend.by,
             var.to.order.obs.by = var.to.order.obs.by,
             var.to.stratify.by = var.to.stratify.by)
  #===================
  # interactnet plot
  #==================
  effect.modifier.var <- "Trx"
  smp.annot <- data.frame(smp.annot,Trx= as.factor(c("Trt","Ctrl")[as.numeric(smp.annot$smp.annot.informative)]))
  
  interactnet.plot(dat = dat,
                   probe.ids = probe.ids,
                   probe.names = probe.names,
                   probe.types = probe.types,
                   smp.annot = smp.annot,
                   effect.modifier.var = effect.modifier.var)
}
